
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <unistd.h>

#include "libfma.h"
#include "lf_dflt_error.h"
#include "lf_alert.h"
#include "lf_fms.h"

#include "insist.h"
#include "fm_cgi_util.h"

#define LONGEST_PATH 1000
#define PATH_SEPARATOR '/'
#define NAMES_FILE "fms_config"


/*
  allocate a big string & reads words from a file into it.
  caller should free the string w/ free ().
*/

static char *
string_from_file(
  char *filename)
{
  FILE *fp;
  int len, c;
  char*s;
  
  assert (filename && *filename);
  
  fp = fopen (filename, "r");
  insistf (fp != NULL, ("couldn't open %s", filename));

  for (len = 0; fgetc (fp) != EOF; len++);
  
  if (len == 0) return 0;

  LF_CALLOC(s, char, len+1);
  
  rewind (fp);

  for (len = 0; (c = fgetc (fp)) != EOF; len++)
    s [len] = c;

  s [len] = 0;
  if (s[len-1] == '\n') s[len-1] = '\0';
  
  fclose (fp);
  
  return s;
  except: return 0;
}

/*
 * connect to fms using fms_config file in cgi dir
 */

int
fm_cgi_connect_to_fms()
{
  char *s;
  int fms;

  s = string_from_file (NAMES_FILE);

  fms = lf_fms_connect (s);
  LF_FREE(s);

  return fms;
}

/*
  search form query string for name=x and return x
*/

char*fm_cgi_get_query_value (char buffer [FM_CGI_MAX_QUERY_VALUE], char*name)
{
  char sn [FM_CGI_MAX_QUERY_VALUE];
  char*s;
  int i;

  insist (buffer && name);
  insistf (strlen (name) < FM_CGI_MAX_QUERY_VALUE - 1, ("name too long"));
  sprintf (sn, "%s=", name);
  
  s = getenv ("QUERY_STRING");
  if (!s) return 0;
  
  if (!(s = strstr (s, sn)))
    return 0;

  s += strlen (sn);
  *buffer = 0;
  for (i = 0; i < FM_CGI_MAX_QUERY_VALUE - 1 && s [i] && s [i] != '&'; i++)
    buffer [i] = s [i];
  buffer [i] = 0;
  
  return buffer;
  except: return 0;
}

/*
  return 1 if name=yes occurs in query string
*/
int fm_cgi_query_yes (char*name)
{
  char buffer [FM_CGI_MAX_QUERY_VALUE];
  char*s;
  
  insist (name);
  
  return (s = fm_cgi_get_query_value (buffer, name)) && !strcmp (s, "yes");

  except: return 0;
}

/*
  read POST data into a struct which caller should free
  w/ fm_cgi_free_form ().
*/

fm_cgi_form_t*fm_cgi_read_form ()
{
  char*s = getenv ("CONTENT_LENGTH");
  int r, c;
  int length = s ? atoi (s) : 0;  
  fm_cgi_form_t*f = calloc (1, sizeof (fm_cgi_form_t));

  
  insistf (f, ("form malloc failed"));
  f->buffer = malloc (length);
  insistf (f->buffer, ("form buffer malloc failed"));

  r = lf_read (STDIN_FILENO, f->buffer, length);
  insistf (r == length, ("reading %d form bytes failed (%d read)", length, r));

  /*count number of tokens so we can alloc arrays*/
  for (s = f->buffer; (s = strchr (s, '=')); s++)
    f->count++;
  
  if (!f->count) return f;

  f->names = calloc (f->count, sizeof (char*));
  f->values = calloc (f->count, sizeof (char*));
  
  insistf (f->names && f->values, ("form array malloc failed"));
  
  /*go back and load up the names / values*/

  for (c = 0, s = strtok (f->buffer, "&=");
       s != NULL && c/2 < f->count;
       s = strtok (0, "&="), c++) {
    if ((c&1) != 0) {
      f->values [c/2] = s;
    } else {
      f->names [c/2] = s;
    }
  }

  insistf (c == f->count * 2, ("unbalanced form data"));
  
  return f;
  except:
  if (f) fm_cgi_free_form (f);
  return 0;
}

/*
  free the struct of form data
*/
void fm_cgi_free_form (fm_cgi_form_t*f)
{
  if (f)
  {
    if (f->buffer) free (f->buffer);
    if (f->names) free (f->names);
    if (f->values) free (f->values);
  }
}

